/*
 * Copyright 2021-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

#include <string.h>
#include <openssl/pkcs7.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/pem.h>
#include "internal/nelem.h"
#include "testutil.h"

#ifndef OPENSSL_NO_EC
static const unsigned char cert_der[] = {
    0x30, 0x82, 0x01, 0x51, 0x30, 0x81, 0xf7, 0xa0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x02, 0x03, 0x09, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x30, 0x27, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
    0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x63, 0x72, 0x79, 0x70, 0x74, 0x6f,
    0x67, 0x72, 0x61, 0x70, 0x68, 0x79, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17,
    0x0d, 0x31, 0x37, 0x30, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x31, 0x30,
    0x30, 0x5a, 0x17, 0x0d, 0x33, 0x38, 0x31, 0x32, 0x33, 0x31, 0x30, 0x38,
    0x33, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x27, 0x31, 0x0b, 0x30, 0x09, 0x06,
    0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x18, 0x30, 0x16,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x63, 0x72, 0x79, 0x70, 0x74,
    0x6f, 0x67, 0x72, 0x61, 0x70, 0x68, 0x79, 0x20, 0x43, 0x41, 0x30, 0x59,
    0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
    0x04, 0x18, 0xff, 0xcf, 0xbb, 0xf9, 0x39, 0xb8, 0xf5, 0xdd, 0xc3, 0xee,
    0xc0, 0x40, 0x8b, 0x06, 0x75, 0x06, 0xab, 0x4f, 0xcd, 0xd8, 0x2c, 0x52,
    0x24, 0x4e, 0x1f, 0xe0, 0x10, 0x46, 0x67, 0xb5, 0x5f, 0x15, 0xb9, 0x62,
    0xbd, 0x3b, 0xcf, 0x0c, 0x6f, 0xbe, 0x1a, 0xf7, 0xb4, 0xa1, 0x0f, 0xb4,
    0xb9, 0xcb, 0x6e, 0x86, 0xb3, 0x50, 0xf9, 0x6c, 0x51, 0xbf, 0xc1, 0x82,
    0xd7, 0xbe, 0xc5, 0xf9, 0x05, 0xa3, 0x13, 0x30, 0x11, 0x30, 0x0f, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01,
    0x01, 0xff, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
    0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xd1, 0x12,
    0xef, 0x8d, 0x97, 0x5a, 0x6e, 0xb8, 0xb6, 0x41, 0xa7, 0xcf, 0xc0, 0xe7,
    0xa4, 0x6e, 0xae, 0xda, 0x51, 0xe4, 0x64, 0x54, 0x2b, 0xde, 0x86, 0x95,
    0xbc, 0xf7, 0x1e, 0x9a, 0xf9, 0x5b, 0x02, 0x21, 0x00, 0xd1, 0x61, 0x86,
    0xce, 0x66, 0x31, 0xe4, 0x2f, 0x54, 0xbd, 0xf5, 0xc8, 0x2b, 0xb3, 0x44,
    0xce, 0x24, 0xf8, 0xa5, 0x0b, 0x72, 0x11, 0x21, 0x34, 0xb9, 0x15, 0x4a,
    0x5f, 0x0e, 0x27, 0x32, 0xa9
};

static const unsigned char smroot_der[] = {
    0x30, 0x82, 0x03, 0x7B, 0x30, 0x82, 0x02, 0x63, 0xA0, 0x03, 0x02, 0x01,
    0x02, 0x02, 0x14, 0x07, 0x18, 0x76, 0x2F, 0x72, 0x2D, 0xB1, 0x53, 0xEE,
    0x06, 0x88, 0x03, 0x23, 0x45, 0x9F, 0x51, 0x7D, 0x65, 0x16, 0x73, 0x30,
    0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B,
    0x05, 0x00, 0x30, 0x44, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
    0x06, 0x13, 0x02, 0x55, 0x4B, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55,
    0x04, 0x0A, 0x0C, 0x0D, 0x4F, 0x70, 0x65, 0x6E, 0x53, 0x53, 0x4C, 0x20,
    0x47, 0x72, 0x6F, 0x75, 0x70, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0C, 0x14, 0x54, 0x65, 0x73, 0x74, 0x20, 0x53, 0x2F, 0x4D,
    0x49, 0x4D, 0x45, 0x20, 0x52, 0x53, 0x41, 0x20, 0x52, 0x6F, 0x6F, 0x74,
    0x30, 0x20, 0x17, 0x0D, 0x32, 0x32, 0x30, 0x36, 0x30, 0x32, 0x31, 0x35,
    0x33, 0x33, 0x31, 0x33, 0x5A, 0x18, 0x0F, 0x32, 0x31, 0x32, 0x32, 0x30,
    0x35, 0x31, 0x30, 0x31, 0x35, 0x33, 0x33, 0x31, 0x33, 0x5A, 0x30, 0x44,
    0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
    0x4B, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0D,
    0x4F, 0x70, 0x65, 0x6E, 0x53, 0x53, 0x4C, 0x20, 0x47, 0x72, 0x6F, 0x75,
    0x70, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x14,
    0x54, 0x65, 0x73, 0x74, 0x20, 0x53, 0x2F, 0x4D, 0x49, 0x4D, 0x45, 0x20,
    0x52, 0x53, 0x41, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x82, 0x01, 0x22,
    0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
    0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A,
    0x02, 0x82, 0x01, 0x01, 0x00, 0xD9, 0x2D, 0x29, 0x7C, 0x2D, 0xD5, 0x39,
    0xE0, 0xE5, 0x00, 0x4F, 0xC7, 0x2D, 0x16, 0xE2, 0x8B, 0x4A, 0x12, 0x6E,
    0x97, 0x62, 0x0C, 0xCD, 0xA1, 0x03, 0x90, 0x52, 0xE8, 0x2F, 0x86, 0xA0,
    0x71, 0xC6, 0xA3, 0x82, 0xFB, 0x11, 0xC5, 0x6F, 0xBC, 0xBD, 0xB7, 0x36,
    0x45, 0xED, 0x32, 0xD6, 0x65, 0x4C, 0xBD, 0xF3, 0x73, 0xEC, 0x28, 0x39,
    0x6B, 0xB2, 0x9E, 0x89, 0x2B, 0x85, 0x24, 0xEA, 0xD0, 0xB6, 0xF0, 0xED,
    0xF8, 0x79, 0x14, 0xE5, 0xCE, 0xF4, 0xE9, 0x80, 0xE6, 0xC3, 0x5E, 0x66,
    0x06, 0x90, 0xD2, 0x1D, 0xAA, 0x64, 0x5E, 0xBF, 0x34, 0x1B, 0xF0, 0x01,
    0x15, 0xBB, 0x2D, 0x59, 0xA7, 0xCE, 0x7D, 0xA8, 0x15, 0x5B, 0x8E, 0x70,
    0xBF, 0xAE, 0x4A, 0x1F, 0x87, 0x27, 0xA5, 0xCA, 0xB0, 0xAC, 0xB8, 0x5A,
    0x82, 0x60, 0xBC, 0x40, 0x6D, 0x0A, 0x7D, 0x66, 0xDF, 0xF9, 0x9D, 0x9E,
    0x7C, 0xEC, 0x5A, 0x1A, 0xEF, 0xDD, 0x4C, 0xD0, 0x37, 0x0A, 0x7A, 0xBF,
    0x05, 0x01, 0x93, 0x76, 0x59, 0x10, 0xDB, 0xCC, 0x5B, 0xAE, 0xF7, 0x7E,
    0x51, 0xBC, 0xE0, 0x71, 0x8E, 0x18, 0xFE, 0x12, 0x7B, 0x17, 0x0D, 0x7F,
    0x21, 0xE1, 0x21, 0x82, 0xBF, 0xE2, 0x54, 0x98, 0x50, 0x21, 0x40, 0x24,
    0x96, 0x9F, 0x41, 0xFB, 0x1A, 0x83, 0x7D, 0x6B, 0xB1, 0x22, 0xDE, 0x66,
    0x37, 0x05, 0x96, 0x87, 0xF2, 0xDF, 0x93, 0xE5, 0x31, 0xA8, 0x4F, 0x7D,
    0x12, 0x14, 0x11, 0xD1, 0x86, 0x56, 0xC8, 0x82, 0x9F, 0x2B, 0x36, 0xDE,
    0x24, 0xB0, 0x73, 0x0E, 0xA6, 0x36, 0x85, 0x0C, 0xD5, 0x2F, 0xA9, 0x00,
    0xBD, 0x8F, 0x5C, 0x47, 0x55, 0x9B, 0x04, 0x8B, 0x53, 0x03, 0x41, 0xB6,
    0x17, 0x9A, 0x99, 0xF4, 0x1F, 0xF7, 0x10, 0x71, 0x05, 0x59, 0x93, 0x01,
    0xBE, 0x86, 0xED, 0xF9, 0x44, 0xC4, 0x92, 0xE0, 0x5D, 0x02, 0x03, 0x01,
    0x00, 0x01, 0xA3, 0x63, 0x30, 0x61, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
    0x0E, 0x04, 0x16, 0x04, 0x14, 0x15, 0xC1, 0x13, 0x21, 0x6B, 0xE2, 0x55,
    0x5F, 0xA7, 0x6A, 0x1D, 0x57, 0x20, 0xD6, 0xCF, 0xF3, 0xA9, 0x73, 0x64,
    0x59, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16,
    0x80, 0x14, 0x15, 0xC1, 0x13, 0x21, 0x6B, 0xE2, 0x55, 0x5F, 0xA7, 0x6A,
    0x1D, 0x57, 0x20, 0xD6, 0xCF, 0xF3, 0xA9, 0x73, 0x64, 0x59, 0x30, 0x0F,
    0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x05, 0x30, 0x03,
    0x01, 0x01, 0xFF, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01,
    0xFF, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x0D, 0x06, 0x09, 0x2A,
    0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03, 0x82,
    0x01, 0x01, 0x00, 0xBD, 0xD0, 0x26, 0xA4, 0x33, 0xE2, 0xD5, 0x6B, 0x7B,
    0x1E, 0x4D, 0xF4, 0x75, 0x72, 0x85, 0xED, 0x48, 0x3A, 0x31, 0x42, 0x84,
    0x4E, 0x2F, 0xAE, 0xA0, 0xED, 0x52, 0xBA, 0x8F, 0x49, 0x84, 0xD9, 0x28,
    0xFD, 0xCA, 0x6F, 0xA2, 0x2F, 0x1F, 0x6E, 0x8F, 0xDD, 0x9D, 0x65, 0x43,
    0x7C, 0x75, 0x68, 0x22, 0xD1, 0x27, 0xEB, 0x8E, 0x12, 0xBE, 0xF7, 0xAB,
    0xE5, 0x42, 0x5D, 0x28, 0x96, 0x3A, 0xD8, 0x3B, 0xE1, 0x8A, 0x5A, 0x42,
    0xB4, 0x0C, 0xB0, 0x61, 0xF0, 0xA9, 0x83, 0x19, 0x29, 0x2A, 0x82, 0x84,
    0x76, 0x7A, 0x63, 0x9D, 0x10, 0x52, 0x31, 0xC6, 0xD5, 0x0C, 0x89, 0xAC,
    0xAA, 0xF7, 0xF5, 0x62, 0x3C, 0xC5, 0x2A, 0x23, 0xA2, 0x0F, 0xB9, 0x1C,
    0x56, 0xFC, 0xF8, 0x57, 0x43, 0x59, 0x63, 0xDD, 0x59, 0x5B, 0x0B, 0xB7,
    0xBE, 0x06, 0x92, 0xBD, 0xC4, 0xC9, 0x42, 0x4A, 0x56, 0x37, 0x46, 0xBE,
    0x2D, 0x66, 0xA7, 0xA5, 0x52, 0xAA, 0x8A, 0x09, 0x7C, 0x9E, 0xC9, 0x2D,
    0xB6, 0x70, 0x5E, 0x88, 0x70, 0xC7, 0x19, 0xE6, 0xF1, 0x8F, 0x78, 0x5A,
    0xC2, 0x52, 0x1D, 0xB5, 0x59, 0x64, 0x6D, 0x0C, 0x0B, 0x2F, 0x09, 0x55,
    0x91, 0x7F, 0x7D, 0xC7, 0x51, 0x7B, 0xDE, 0xE4, 0xB7, 0xDC, 0x8C, 0xC1,
    0x63, 0xAF, 0x99, 0x27, 0x68, 0xAC, 0xEE, 0x60, 0xFC, 0xDF, 0xD0, 0x7C,
    0x7C, 0xE5, 0x5D, 0xF6, 0x9B, 0xCF, 0xBC, 0x10, 0x60, 0x6A, 0x12, 0x2B,
    0xC3, 0x9B, 0x73, 0x27, 0xF4, 0x3E, 0x80, 0x84, 0xEE, 0x55, 0x07, 0x7D,
    0x13, 0xA4, 0x55, 0x5E, 0x1D, 0x74, 0xFC, 0x0C, 0x3E, 0x8B, 0xFF, 0x12,
    0xEE, 0xD1, 0xE9, 0x11, 0x48, 0x0F, 0xDE, 0x10, 0x64, 0xD4, 0xFF, 0xB0,
    0x45, 0x00, 0x56, 0x0F, 0x16, 0xC7, 0x07, 0x6F, 0x25, 0x13, 0xEE, 0x35,
    0x8E, 0xF8, 0x77, 0x24, 0xE6, 0xD0, 0x59
};

static int pkcs7_verify_test(void)
{
    int ret = 0;
    size_t i;
    BIO *msg_bio = NULL, *x509_bio = NULL, *bio = NULL;
    X509 *cert = NULL;
    X509_STORE *store = NULL;
    PKCS7 *p7 = NULL;
    const char *sig[] = {
        "MIME-Version: 1.0\nContent-Type: multipart/signed; protocol=\"application/x-pkcs7-signature\"; micalg=\"sha-256\"; boundary=\"----9B5319FF2E4428B17CD26B69294E7F31\"\n\n",
        "This is an S/MIME signed message\n\n------9B5319FF2E4428B17CD26B69294E7F31\n",
        "Content-Type: text/plain\r\n\r\nhello world\n------9B5319FF2E4428B17CD26B69294E7F31\n",
        "Content-Type: application/x-pkcs7-signature; name=\"smime.p7s\"\n",
        "Content-Transfer-Encoding: base64\nContent-Disposition: attachment; filename=\"smime.p7s\"\n\n",
        "MIIDEgYJKoZIhvcNAQcCoIIDAzCCAv8CAQExDzANBglghkgBZQMEAgEFADALBgkq\nhkiG9w0BBwGgggFVMIIBUTCB96ADAgECAgIDCTAKBggqhkjOPQQDAjAnMQswCQYD\nVQQGEwJVUzEYMBYGA1UEAwwPY3J5cHRvZ3JhcGh5IENBMB4XDTE3MDEwMTEyMDEw\nMFoXDTM4MTIzMTA4MzAwMFowJzELMAkGA1UEBhMCVVMxGDAWBgNVBAMMD2NyeXB0\nb2dyYXBoeSBDQTBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABBj/z7v5Obj13cPu\nwECLBnUGq0/N2CxSJE4f4BBGZ7VfFblivTvPDG++Gve0oQ+0uctuhrNQ+WxRv8GC\n",
        "177F+QWjEzARMA8GA1UdEwEB/wQFMAMBAf8wCgYIKoZIzj0EAwIDSQAwRgIhANES\n742XWm64tkGnz8DnpG6u2lHkZFQr3oaVvPcemvlbAiEA0WGGzmYx5C9UvfXIK7NE\nziT4pQtyESE0uRVKXw4nMqkxggGBMIIBfQIBATAtMCcxCzAJBgNVBAYTAlVTMRgw\nFgYDVQQDDA9jcnlwdG9ncmFwaHkgQ0ECAgMJMA0GCWCGSAFlAwQCAQUAoIHkMBgG\nCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIxMDUyMDE4\nNTA0OVowLwYJKoZIhvcNAQkEMSIEIOdwMRgQrqcnmMYvag+BVvErcc6bwUXI94Ds\n",
        "QkiyIU9pMHkGCSqGSIb3DQEJDzFsMGowCwYJYIZIAWUDBAEqMAsGCWCGSAFlAwQB\nFjALBglghkgBZQMEAQIwCgYIKoZIhvcNAwcwDgYIKoZIhvcNAwICAgCAMA0GCCqG\nSIb3DQMCAgFAMAcGBSsOAwIHMA0GCCqGSIb3DQMCAgEoMAoGCCqGSM49BAMCBEcw\nRQIhANYMJku1fW9T1MIEcAyREArz9kXCY4tWck5Pt0xzrYhaAiBDSP6e43zj4YtI\nuvQW+Lzv+dNF8EPuhgoPNe17RuUSLw==\n\n------9B5319FF2E4428B17CD26B69294E7F31--\n\n"
    };
    const char *signed_data = "Content-Type: text/plain\r\n\r\nhello world";

    if (!TEST_ptr(bio = BIO_new(BIO_s_mem())))
        goto end;
    for  (i = 0; i < OSSL_NELEM(sig); ++i)
        BIO_puts(bio, sig[i]);

    ret = TEST_ptr(msg_bio = BIO_new_mem_buf(signed_data, (int)strlen(signed_data)))
          && TEST_ptr(x509_bio = BIO_new_mem_buf(cert_der, sizeof(cert_der)))
          && TEST_ptr(cert = d2i_X509_bio(x509_bio, NULL))
          && TEST_int_eq(ERR_peek_error(), 0)
          && TEST_ptr(store = X509_STORE_new())
          && TEST_true(X509_STORE_add_cert(store, cert))
          && TEST_ptr(p7 = SMIME_read_PKCS7(bio, NULL))
          && TEST_int_eq(ERR_peek_error(), 0)
          && TEST_true(PKCS7_verify(p7, NULL, store, msg_bio, NULL, PKCS7_TEXT))
          && TEST_int_eq(ERR_peek_error(), 0);
end:
    X509_STORE_free(store);
    X509_free(cert);
    PKCS7_free(p7);
    BIO_free(msg_bio);
    BIO_free(x509_bio);
    BIO_free(bio);
    return ret;
}

/*
 * PKCS7_verify() BIO *indata parameter refers to the signed data if the content
 * is detached from p7. Otherwise indata should be NULL, and then the signed
 * data must be in p7. The content is written to the BIO out unless it is NULL.
 *
 * $ openssl asn1parse -i -inform pem -in sig.pem
 *    0:d=0  hl=4 l=1571 cons: SEQUENCE
 *    4:d=1  hl=2 l=   9 prim:  OBJECT            :pkcs7-signedData
 *   15:d=1  hl=4 l=1556 cons:  cont [ 0 ]
 *   19:d=2  hl=4 l=1552 cons:   SEQUENCE
 *** ASN1_INTEGER *version; ***
 *   23:d=3  hl=2 l=   1 prim:    INTEGER           :01
 *** STACK_OF(X509_ALGOR) *md_algs;
 *   26:d=3  hl=2 l=  15 cons:    SET
 *   28:d=4  hl=2 l=  13 cons:     SEQUENCE
 *   30:d=5  hl=2 l=   9 prim:      OBJECT            :sha256
 *   41:d=5  hl=2 l=   0 prim:      NULL
 *** struct pkcs7_st *contents; ***
 *   43:d=3  hl=2 l=  92 cons:    SEQUENCE
 *   45:d=4  hl=2 l=  10 prim:     OBJECT            :1.3.6.1.4.1.311.2.1.4
 *   57:d=4  hl=2 l=  78 cons:     cont [ 0 ]
 *   59:d=5  hl=2 l=  76 cons:      SEQUENCE
 *   61:d=6  hl=2 l=  23 cons:       SEQUENCE
 *   63:d=7  hl=2 l=  10 prim:        OBJECT            :1.3.6.1.4.1.311.2.1.15
 *   75:d=7  hl=2 l=   9 cons:        SEQUENCE
 *   77:d=8  hl=2 l=   1 prim:         BIT STRING
 *   80:d=8  hl=2 l=   4 cons:         cont [ 0 ]
 *   82:d=9  hl=2 l=   2 cons:          cont [ 2 ]
 *   84:d=10 hl=2 l=   0 prim:           cont [ 0 ]
 *   86:d=6  hl=2 l=  49 cons:       SEQUENCE
 *   88:d=7  hl=2 l=  13 cons:        SEQUENCE
 *   90:d=8  hl=2 l=   9 prim:         OBJECT            :sha256
 *  101:d=8  hl=2 l=   0 prim:         NULL
 *** Hash of a signed PE file (sha256) ***
 *  103:d=7  hl=2 l=  32 prim:        OCTET STRING
 *   [HEX DUMP]:2D2C7B382C8163A419B9FF214A7B651C33F9EA43335907F11377290C5158A7A4
 *** STACK_OF(X509) *cert; ***
 *  137:d=3  hl=4 l= 913 cons:    cont [ 0 ]
 *  141:d=4  hl=4 l= 909 cons:     SEQUENCE
 *  145:d=5  hl=4 l= 629 cons:      SEQUENCE
 *  ...
 *** STACK_OF(PKCS7_SIGNER_INFO) *signer_info; ***
 * 1054:d=3  hl=4 l= 517 cons:    SET
 * 1058:d=4  hl=4 l= 513 cons:     SEQUENCE
 * ...
 */

static int pkcs7_inner_content_verify_test(void)
{
    int ret = 0;
    BIO *x509_bio = NULL, *bio = NULL;
    X509 *cert = NULL;
    X509_STORE *store = NULL;
    PKCS7 *p7 = NULL;
    X509_VERIFY_PARAM *param;
    const unsigned char sig_der[] = {
        0x30, 0x82, 0x06, 0x23, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
        0x01, 0x07, 0x02, 0xA0, 0x82, 0x06, 0x14, 0x30, 0x82, 0x06, 0x10, 0x02,
        0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
        0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x5C, 0x06, 0x0A, 0x2B,
        0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x04, 0xA0, 0x4E, 0x30,
        0x4C, 0x30, 0x17, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
        0x02, 0x01, 0x0F, 0x30, 0x09, 0x03, 0x01, 0x00, 0xA0, 0x04, 0xA2, 0x02,
        0x80, 0x00, 0x30, 0x31, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
        0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x04, 0x20, 0x2D, 0x2C, 0x7B,
        0x38, 0x2C, 0x81, 0x63, 0xA4, 0x19, 0xB9, 0xFF, 0x21, 0x4A, 0x7B, 0x65,
        0x1C, 0x33, 0xF9, 0xEA, 0x43, 0x33, 0x59, 0x07, 0xF1, 0x13, 0x77, 0x29,
        0x0C, 0x51, 0x58, 0xA7, 0xA4, 0xA0, 0x82, 0x03, 0x91, 0x30, 0x82, 0x03,
        0x8D, 0x30, 0x82, 0x02, 0x75, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x14,
        0x1A, 0x2F, 0x94, 0x5F, 0x4D, 0x1E, 0x9B, 0x68, 0xF8, 0xBF, 0xC6, 0xC9,
        0xD6, 0xC7, 0x07, 0xB0, 0x3E, 0x35, 0xFD, 0xC7, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30,
        0x44, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
        0x55, 0x4B, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C,
        0x0D, 0x4F, 0x70, 0x65, 0x6E, 0x53, 0x53, 0x4C, 0x20, 0x47, 0x72, 0x6F,
        0x75, 0x70, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
        0x14, 0x54, 0x65, 0x73, 0x74, 0x20, 0x53, 0x2F, 0x4D, 0x49, 0x4D, 0x45,
        0x20, 0x52, 0x53, 0x41, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x20, 0x17,
        0x0D, 0x32, 0x32, 0x30, 0x38, 0x31, 0x36, 0x31, 0x31, 0x35, 0x38, 0x30,
        0x30, 0x5A, 0x18, 0x0F, 0x32, 0x31, 0x32, 0x32, 0x30, 0x38, 0x31, 0x36,
        0x31, 0x31, 0x35, 0x38, 0x30, 0x30, 0x5A, 0x30, 0x47, 0x31, 0x0B, 0x30,
        0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x4B, 0x31, 0x16,
        0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0D, 0x4F, 0x70, 0x65,
        0x6E, 0x53, 0x53, 0x4C, 0x20, 0x47, 0x72, 0x6F, 0x75, 0x70, 0x31, 0x20,
        0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73,
        0x74, 0x20, 0x43, 0x6F, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x45,
        0x45, 0x20, 0x52, 0x53, 0x41, 0x20, 0x23, 0x31, 0x30, 0x82, 0x01, 0x22,
        0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
        0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A,
        0x02, 0x82, 0x01, 0x01, 0x00, 0xA8, 0xFF, 0x89, 0x58, 0x61, 0x85, 0xAE,
        0xDE, 0xDB, 0x2C, 0x0F, 0xD5, 0x73, 0xFB, 0xE2, 0xB7, 0xA8, 0x1B, 0x7A,
        0x75, 0x60, 0x2B, 0xD4, 0x7D, 0x3C, 0x73, 0x76, 0xB5, 0xAE, 0xDE, 0x93,
        0x2F, 0x7F, 0x24, 0x90, 0x6D, 0xC9, 0x1C, 0xF4, 0x8A, 0x38, 0xCD, 0x9F,
        0xA2, 0x80, 0x16, 0x56, 0xCB, 0xB4, 0x2C, 0xBB, 0x93, 0xE5, 0x17, 0x1F,
        0xFF, 0x0D, 0xC6, 0x43, 0xFE, 0x31, 0x34, 0xF8, 0x28, 0xD8, 0x75, 0xE9,
        0xA3, 0xBB, 0x88, 0x82, 0xEF, 0xA3, 0x72, 0xE2, 0x53, 0x62, 0x05, 0x45,
        0x4D, 0xE0, 0x31, 0x97, 0xEF, 0xAA, 0x9D, 0x10, 0xD2, 0x58, 0x0D, 0x9E,
        0x06, 0x9E, 0x32, 0xBB, 0x1B, 0x08, 0x56, 0x5D, 0x65, 0x0F, 0x3F, 0x66,
        0x8E, 0xC2, 0x36, 0xA0, 0x11, 0xFF, 0xC0, 0x59, 0x35, 0xFD, 0x49, 0x1C,
        0xDD, 0x1C, 0x42, 0x33, 0x06, 0x78, 0xAE, 0x04, 0x38, 0x23, 0xE5, 0xE7,
        0x75, 0x67, 0xD3, 0x94, 0xAD, 0x5A, 0xA9, 0x40, 0x1F, 0xAC, 0x59, 0x10,
        0xF5, 0x0C, 0xC1, 0x64, 0xAD, 0x4A, 0xC4, 0xC6, 0x35, 0x53, 0xB3, 0x96,
        0x6A, 0x82, 0x1B, 0x15, 0x8B, 0xD1, 0x99, 0xAB, 0xE4, 0x9E, 0x6F, 0x94,
        0xAA, 0xD2, 0x96, 0xE5, 0x57, 0xFB, 0x8C, 0x91, 0x64, 0x8A, 0x3C, 0x24,
        0x16, 0xC6, 0xD2, 0x19, 0xBE, 0x82, 0x60, 0x13, 0x3B, 0x62, 0xB0, 0xED,
        0x3E, 0x1D, 0xB3, 0xCA, 0xF9, 0x6B, 0xDF, 0x31, 0xB9, 0x7A, 0x9E, 0x4B,
        0x47, 0x68, 0xA7, 0x29, 0xC7, 0x3F, 0x4A, 0xE2, 0x22, 0x27, 0xB4, 0xEF,
        0x90, 0x63, 0xD7, 0xEF, 0xD0, 0x72, 0x49, 0x3E, 0x5B, 0xAD, 0xB9, 0x1B,
        0xD4, 0x2B, 0x6F, 0x86, 0x68, 0xDD, 0xD5, 0x73, 0x12, 0xB8, 0x43, 0xC9,
        0xDC, 0x41, 0x0F, 0xA2, 0xA3, 0x46, 0x4F, 0x8B, 0x67, 0x88, 0x84, 0x0F,
        0x61, 0xFE, 0x79, 0x34, 0x1F, 0x4A, 0x89, 0x1F, 0xB7, 0x02, 0x03, 0x01,
        0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1D,
        0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F,
        0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x13, 0x06,
        0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06,
        0x01, 0x05, 0x05, 0x07, 0x03, 0x03, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
        0x0E, 0x04, 0x16, 0x04, 0x14, 0xE7, 0x9B, 0xE2, 0x2A, 0xAD, 0x8A, 0x6C,
        0x3A, 0xCB, 0x76, 0x51, 0xE5, 0x8E, 0x07, 0x98, 0x22, 0x97, 0xE1, 0x73,
        0xA2, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16,
        0x80, 0x14, 0x15, 0xC1, 0x13, 0x21, 0x6B, 0xE2, 0x55, 0x5F, 0xA7, 0x6A,
        0x1D, 0x57, 0x20, 0xD6, 0xCF, 0xF3, 0xA9, 0x73, 0x64, 0x59, 0x30, 0x0D,
        0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05,
        0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x32, 0x92, 0xC7, 0x7B, 0x6A, 0xB7,
        0x39, 0x82, 0x7C, 0x90, 0xA6, 0x00, 0xBE, 0x34, 0xB9, 0x54, 0x38, 0x84,
        0x4A, 0xA3, 0xCF, 0x57, 0x26, 0x0C, 0x48, 0xA6, 0xFA, 0x07, 0xB6, 0xEC,
        0x7B, 0x1F, 0xC1, 0x80, 0x50, 0x2B, 0xC6, 0x69, 0x3E, 0xF2, 0x13, 0xA9,
        0xBB, 0x2B, 0xAA, 0x4A, 0x87, 0xA3, 0x8B, 0x25, 0x40, 0x3B, 0xDA, 0xDE,
        0xFC, 0x9A, 0xE6, 0x41, 0xEB, 0x4C, 0xD6, 0xD3, 0x68, 0xEE, 0x47, 0x7D,
        0x47, 0x75, 0x83, 0x2F, 0x50, 0xF7, 0xE2, 0x11, 0xBB, 0x92, 0x00, 0xD3,
        0x01, 0xAA, 0x9B, 0x70, 0x13, 0xC3, 0x51, 0xCE, 0xE0, 0x8A, 0x7F, 0x1B,
        0x4E, 0x46, 0x77, 0x4C, 0x9E, 0x9B, 0xED, 0xC5, 0x09, 0xAF, 0x08, 0x7D,
        0x5C, 0x2C, 0x16, 0x5D, 0x37, 0x0E, 0x94, 0x9D, 0x4E, 0xE7, 0x05, 0xDE,
        0x17, 0xD9, 0x80, 0x4F, 0x21, 0xD9, 0x1B, 0x00, 0xA1, 0x3A, 0xBA, 0xE8,
        0x88, 0x4C, 0xC3, 0x91, 0x1A, 0x0E, 0x75, 0xE3, 0xE3, 0xD2, 0xC2, 0x8D,
        0x65, 0x94, 0x68, 0xA6, 0x33, 0x95, 0x67, 0x9D, 0x1D, 0xFD, 0x7F, 0x09,
        0x30, 0x29, 0x96, 0xFF, 0x1F, 0x25, 0xBC, 0xEB, 0xA2, 0x47, 0xEB, 0x83,
        0x4E, 0x8B, 0x47, 0xF7, 0xEB, 0x1B, 0xDE, 0xD8, 0x21, 0x27, 0x47, 0x26,
        0xA5, 0x52, 0xA4, 0x14, 0x2D, 0x29, 0x5D, 0x2F, 0xF4, 0x0C, 0x1E, 0x6A,
        0x54, 0x7A, 0xE8, 0x84, 0x97, 0x64, 0xC0, 0xB4, 0x6E, 0xF0, 0x05, 0xF7,
        0x09, 0x66, 0xDC, 0x42, 0x01, 0xD6, 0x83, 0xB0, 0x51, 0x65, 0xB4, 0x3A,
        0x4D, 0xA9, 0x90, 0x07, 0xC0, 0x25, 0x5D, 0xD6, 0x23, 0xF0, 0x5B, 0x3B,
        0x9B, 0xB0, 0xCC, 0x92, 0x49, 0x81, 0xC6, 0x3A, 0xD0, 0x52, 0xED, 0xEC,
        0xEC, 0x54, 0x53, 0x00, 0xBC, 0x69, 0xFE, 0x5A, 0x8C, 0x72, 0x86, 0x99,
        0xCB, 0xE6, 0xDD, 0x07, 0x37, 0x79, 0xF4, 0x66, 0x8E, 0x15, 0x31, 0x82,
        0x02, 0x05, 0x30, 0x82, 0x02, 0x01, 0x02, 0x01, 0x01, 0x30, 0x5C, 0x30,
        0x44, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
        0x55, 0x4B, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C,
        0x0D, 0x4F, 0x70, 0x65, 0x6E, 0x53, 0x53, 0x4C, 0x20, 0x47, 0x72, 0x6F,
        0x75, 0x70, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
        0x14, 0x54, 0x65, 0x73, 0x74, 0x20, 0x53, 0x2F, 0x4D, 0x49, 0x4D, 0x45,
        0x20, 0x52, 0x53, 0x41, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x02, 0x14, 0x1A,
        0x2F, 0x94, 0x5F, 0x4D, 0x1E, 0x9B, 0x68, 0xF8, 0xBF, 0xC6, 0xC9, 0xD6,
        0xC7, 0x07, 0xB0, 0x3E, 0x35, 0xFD, 0xC7, 0x30, 0x0D, 0x06, 0x09, 0x60,
        0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA0, 0x7C,
        0x30, 0x10, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02,
        0x01, 0x0C, 0x31, 0x02, 0x30, 0x00, 0x30, 0x19, 0x06, 0x09, 0x2A, 0x86,
        0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x03, 0x31, 0x0C, 0x06, 0x0A, 0x2B,
        0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x04, 0x30, 0x1C, 0x06,
        0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x0B, 0x31,
        0x0E, 0x30, 0x0C, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
        0x02, 0x01, 0x15, 0x30, 0x2F, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
        0x0D, 0x01, 0x09, 0x04, 0x31, 0x22, 0x04, 0x20, 0x06, 0x1E, 0x2B, 0xC2,
        0x31, 0xF7, 0x1B, 0xA4, 0x8C, 0x65, 0x4A, 0x26, 0x5B, 0xAD, 0x3E, 0x8C,
        0x25, 0xB3, 0xAA, 0xAF, 0xB2, 0x0F, 0xA1, 0xF6, 0x8B, 0x6A, 0x6B, 0x87,
        0x45, 0x47, 0x9E, 0x35, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0x45,
        0x98, 0xCB, 0xA9, 0x2B, 0x49, 0xFD, 0x33, 0xA0, 0xBD, 0x27, 0x98, 0xE1,
        0xA4, 0x2D, 0xD7, 0x73, 0xC1, 0x77, 0x66, 0x97, 0x26, 0x5A, 0xA0, 0x72,
        0xD9, 0x8E, 0x3B, 0x9B, 0x74, 0x20, 0xCF, 0x13, 0x6A, 0xE1, 0xD3, 0x97,
        0xAC, 0xAB, 0xEA, 0x4A, 0x43, 0x98, 0xA2, 0x7C, 0x71, 0x95, 0xFF, 0x61,
        0x7B, 0x9B, 0x56, 0xF7, 0xBF, 0xF0, 0x33, 0xB4, 0x3E, 0xBF, 0xF0, 0x8A,
        0xA0, 0x78, 0x3B, 0xF9, 0xAF, 0x9E, 0xFD, 0x54, 0x92, 0xC5, 0x81, 0x6D,
        0xE0, 0x29, 0x78, 0xBC, 0x00, 0xD8, 0xF1, 0xA6, 0xFF, 0x00, 0x03, 0xF9,
        0x8E, 0x9E, 0xB4, 0x09, 0x10, 0x4A, 0x1D, 0xF6, 0x0E, 0x76, 0x15, 0xAA,
        0x96, 0x86, 0x38, 0x3A, 0x0B, 0x3C, 0x0E, 0x37, 0x6E, 0xF9, 0xA3, 0x59,
        0x0F, 0x41, 0x42, 0xC3, 0xA2, 0x74, 0x77, 0x10, 0x7F, 0xA0, 0xA8, 0x17,
        0xFB, 0xD2, 0x68, 0x8A, 0x17, 0x71, 0x89, 0x2C, 0x30, 0x49, 0x54, 0xDF,
        0x29, 0x80, 0xC2, 0xB9, 0x13, 0x36, 0x77, 0xF4, 0xBE, 0x6B, 0x76, 0xBC,
        0x42, 0x1A, 0xB2, 0xE7, 0x4D, 0x36, 0xED, 0x85, 0xB8, 0x25, 0xEC, 0x03,
        0xF8, 0x1E, 0x2F, 0x41, 0x48, 0x18, 0xAD, 0x58, 0x3D, 0x40, 0xEB, 0x4F,
        0x5B, 0xFB, 0x15, 0x57, 0xD5, 0xE0, 0x68, 0xC6, 0x84, 0x21, 0x57, 0x9E,
        0x20, 0x46, 0x45, 0xDB, 0x30, 0xA3, 0x6F, 0x44, 0x35, 0x17, 0xAC, 0xE6,
        0xD1, 0x0E, 0xEE, 0x43, 0x38, 0x87, 0xCD, 0x22, 0xE9, 0x83, 0xFE, 0x9D,
        0x30, 0x8F, 0xE0, 0x51, 0xDB, 0xFF, 0x26, 0x5A, 0x53, 0x21, 0xBF, 0xE7,
        0x1F, 0xD8, 0xA4, 0x53, 0xDD, 0xAE, 0xF2, 0xF5, 0x73, 0xEA, 0xFE, 0x50,
        0x99, 0x1B, 0x16, 0x1C, 0xBD, 0x4E, 0xFD, 0x8B, 0x2E, 0x22, 0x31, 0x05,
        0x90, 0x9D, 0x41, 0x7C, 0xA7, 0x83, 0xE9, 0xCA, 0x8E, 0x01, 0x6D, 0xB2,
        0x2D, 0x6F, 0x81
    };

    if (!TEST_ptr(bio = BIO_new_mem_buf(sig_der, sizeof sig_der)))
        goto end;

    ret = TEST_ptr(x509_bio = BIO_new_mem_buf(smroot_der, sizeof smroot_der))
            && TEST_ptr(cert = d2i_X509_bio(x509_bio, NULL))
            && TEST_int_eq(ERR_peek_error(), 0)
            && TEST_ptr(store = X509_STORE_new())
            && TEST_true(X509_STORE_add_cert(store, cert))
            && TEST_ptr(param = X509_STORE_get0_param(store))
            && TEST_true(X509_VERIFY_PARAM_set_purpose(param,
                                                       X509_PURPOSE_CODE_SIGN))
            && TEST_true(X509_STORE_set1_param(store, param))
            && TEST_ptr(p7 = d2i_PKCS7_bio(bio, NULL))
            && TEST_int_eq(ERR_peek_error(), 0)
            && TEST_true(PKCS7_verify(p7, NULL, store, NULL, NULL, 0))
            && TEST_int_eq(ERR_peek_error(), 0);
end:
    X509_STORE_free(store);
    X509_free(cert);
    PKCS7_free(p7);
    BIO_free(x509_bio);
    BIO_free(bio);
    return ret;
}
#endif /* OPENSSL_NO_EC */

int setup_tests(void)
{
#ifndef OPENSSL_NO_EC
    ADD_TEST(pkcs7_verify_test);
    ADD_TEST(pkcs7_inner_content_verify_test);
#endif /* OPENSSL_NO_EC */
    return 1;
}
